/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import java.util.Arrays;
import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.exception.util.LocalizedFormats;
import org.apache.commons.math.linear.ArrayRealVector;
import org.apache.commons.math.linear.BlockRealMatrix;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.QRDecomposition;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.util.FastMath;

public class QRDecompositionImpl
implements QRDecomposition {
    private double[][] qrt;
    private double[] rDiag;
    private RealMatrix cachedQ;
    private RealMatrix cachedQT;
    private RealMatrix cachedR;
    private RealMatrix cachedH;

    public QRDecompositionImpl(RealMatrix matrix2) {
        int m = matrix2.getRowDimension();
        int n = matrix2.getColumnDimension();
        this.qrt = matrix2.transpose().getData();
        this.rDiag = new double[FastMath.min(m, n)];
        this.cachedQ = null;
        this.cachedQT = null;
        this.cachedR = null;
        this.cachedH = null;
        for (int minor = 0; minor < FastMath.min(m, n); ++minor) {
            double a;
            double[] qrtMinor = this.qrt[minor];
            double xNormSqr = 0.0;
            for (int row2 = minor; row2 < m; ++row2) {
                double c2 = qrtMinor[row2];
                xNormSqr += c2 * c2;
            }
            this.rDiag[minor] = a = qrtMinor[minor] > 0.0 ? -FastMath.sqrt(xNormSqr) : FastMath.sqrt(xNormSqr);
            if (a == 0.0) continue;
            int n2 = minor;
            qrtMinor[n2] = qrtMinor[n2] - a;
            for (int col2 = minor + 1; col2 < n; ++col2) {
                int row3;
                double[] qrtCol = this.qrt[col2];
                double alpha = 0.0;
                for (row3 = minor; row3 < m; ++row3) {
                    alpha -= qrtCol[row3] * qrtMinor[row3];
                }
                alpha /= a * qrtMinor[minor];
                for (row3 = minor; row3 < m; ++row3) {
                    int n3 = row3;
                    qrtCol[n3] = qrtCol[n3] - alpha * qrtMinor[row3];
                }
            }
        }
    }

    public RealMatrix getR() {
        if (this.cachedR == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedR = MatrixUtils.createRealMatrix(m, n);
            for (int row2 = FastMath.min(m, n) - 1; row2 >= 0; --row2) {
                this.cachedR.setEntry(row2, row2, this.rDiag[row2]);
                for (int col2 = row2 + 1; col2 < n; ++col2) {
                    this.cachedR.setEntry(row2, col2, this.qrt[col2][row2]);
                }
            }
        }
        return this.cachedR;
    }

    public RealMatrix getQ() {
        if (this.cachedQ == null) {
            this.cachedQ = this.getQT().transpose();
        }
        return this.cachedQ;
    }

    public RealMatrix getQT() {
        if (this.cachedQT == null) {
            int minor;
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedQT = MatrixUtils.createRealMatrix(m, m);
            for (minor = m - 1; minor >= FastMath.min(m, n); --minor) {
                this.cachedQT.setEntry(minor, minor, 1.0);
            }
            for (minor = FastMath.min(m, n) - 1; minor >= 0; --minor) {
                double[] qrtMinor = this.qrt[minor];
                this.cachedQT.setEntry(minor, minor, 1.0);
                if (qrtMinor[minor] == 0.0) continue;
                for (int col2 = minor; col2 < m; ++col2) {
                    int row2;
                    double alpha = 0.0;
                    for (row2 = minor; row2 < m; ++row2) {
                        alpha -= this.cachedQT.getEntry(col2, row2) * qrtMinor[row2];
                    }
                    alpha /= this.rDiag[minor] * qrtMinor[minor];
                    for (row2 = minor; row2 < m; ++row2) {
                        this.cachedQT.addToEntry(col2, row2, -alpha * qrtMinor[row2]);
                    }
                }
            }
        }
        return this.cachedQT;
    }

    public RealMatrix getH() {
        if (this.cachedH == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedH = MatrixUtils.createRealMatrix(m, n);
            for (int i = 0; i < m; ++i) {
                for (int j = 0; j < FastMath.min(i + 1, n); ++j) {
                    this.cachedH.setEntry(i, j, this.qrt[j][i] / -this.rDiag[j]);
                }
            }
        }
        return this.cachedH;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.qrt, this.rDiag);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] qrt;
        private final double[] rDiag;

        private Solver(double[][] qrt, double[] rDiag) {
            this.qrt = qrt;
            this.rDiag = rDiag;
        }

        public boolean isNonSingular() {
            for (double diag : this.rDiag) {
                if (diag != 0.0) continue;
                return false;
            }
            return true;
        }

        public double[] solve(double[] b) throws IllegalArgumentException, InvalidMatrixException {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b.length != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b.length, m);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            double[] x = new double[n];
            double[] y = (double[])b.clone();
            for (int minor = 0; minor < FastMath.min(m, n); ++minor) {
                int row2;
                double[] qrtMinor = this.qrt[minor];
                double dotProduct = 0.0;
                for (row2 = minor; row2 < m; ++row2) {
                    dotProduct += y[row2] * qrtMinor[row2];
                }
                dotProduct /= this.rDiag[minor] * qrtMinor[minor];
                for (row2 = minor; row2 < m; ++row2) {
                    int n2 = row2;
                    y[n2] = y[n2] + dotProduct * qrtMinor[row2];
                }
            }
            for (int row3 = this.rDiag.length - 1; row3 >= 0; --row3) {
                int n3 = row3;
                y[n3] = y[n3] / this.rDiag[row3];
                double yRow = y[row3];
                double[] qrtRow = this.qrt[row3];
                x[row3] = yRow;
                for (int i = 0; i < row3; ++i) {
                    int n4 = i;
                    y[n4] = y[n4] - yRow * qrtRow[i];
                }
            }
            return x;
        }

        public RealVector solve(RealVector b) throws IllegalArgumentException, InvalidMatrixException {
            try {
                return this.solve((ArrayRealVector)b);
            }
            catch (ClassCastException cce) {
                return new ArrayRealVector(this.solve(b.getData()), false);
            }
        }

        public ArrayRealVector solve(ArrayRealVector b) throws IllegalArgumentException, InvalidMatrixException {
            return new ArrayRealVector(this.solve(b.getDataRef()), false);
        }

        public RealMatrix solve(RealMatrix b) throws IllegalArgumentException, InvalidMatrixException {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b.getRowDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.DIMENSIONS_MISMATCH_2x2, b.getRowDimension(), b.getColumnDimension(), m, "n");
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int columns = b.getColumnDimension();
            int blockSize = 52;
            int cBlocks = (columns + 52 - 1) / 52;
            double[][] xBlocks = BlockRealMatrix.createBlocksLayout(n, columns);
            double[][] y = new double[b.getRowDimension()][52];
            double[] alpha = new double[52];
            for (int kBlock = 0; kBlock < cBlocks; ++kBlock) {
                int kStart = kBlock * 52;
                int kEnd = FastMath.min(kStart + 52, columns);
                int kWidth = kEnd - kStart;
                b.copySubMatrix(0, m - 1, kStart, kEnd - 1, y);
                for (int minor = 0; minor < FastMath.min(m, n); ++minor) {
                    int k;
                    double[] yRow;
                    double d;
                    int row2;
                    double[] qrtMinor = this.qrt[minor];
                    double factor = 1.0 / (this.rDiag[minor] * qrtMinor[minor]);
                    Arrays.fill(alpha, 0, kWidth, 0.0);
                    for (row2 = minor; row2 < m; ++row2) {
                        d = qrtMinor[row2];
                        yRow = y[row2];
                        for (k = 0; k < kWidth; ++k) {
                            int n2 = k;
                            alpha[n2] = alpha[n2] + d * yRow[k];
                        }
                    }
                    int k2 = 0;
                    while (k2 < kWidth) {
                        int n3 = k2++;
                        alpha[n3] = alpha[n3] * factor;
                    }
                    for (row2 = minor; row2 < m; ++row2) {
                        d = qrtMinor[row2];
                        yRow = y[row2];
                        for (k = 0; k < kWidth; ++k) {
                            int n4 = k;
                            yRow[n4] = yRow[n4] + alpha[k] * d;
                        }
                    }
                }
                for (int j = this.rDiag.length - 1; j >= 0; --j) {
                    int jBlock = j / 52;
                    int jStart = jBlock * 52;
                    double factor = 1.0 / this.rDiag[j];
                    double[] yJ = y[j];
                    double[] xBlock = xBlocks[jBlock * cBlocks + kBlock];
                    int index = (j - jStart) * kWidth;
                    for (int k = 0; k < kWidth; ++k) {
                        int n5 = k;
                        yJ[n5] = yJ[n5] * factor;
                        xBlock[index++] = yJ[k];
                    }
                    double[] qrtJ = this.qrt[j];
                    for (int i = 0; i < j; ++i) {
                        double rIJ = qrtJ[i];
                        double[] yI = y[i];
                        for (int k = 0; k < kWidth; ++k) {
                            int n6 = k;
                            yI[n6] = yI[n6] - yJ[k] * rIJ;
                        }
                    }
                }
            }
            return new BlockRealMatrix(n, columns, xBlocks, false);
        }

        public RealMatrix getInverse() throws InvalidMatrixException {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.rDiag.length));
        }
    }
}

